#include "program.h"
//#include <chrono>

namespace lp {

	std::ostream& operator<<(std::ostream& os, const Program& p)
	{
		if (p.mode_begin() != p.mode_end()) {
			os << "Mode Declarations:\n";
			for (auto m = p.mode_begin(); m != p.mode_end(); ++m) {
				os << "  " << *m << "\n";
			}
		}

		for (auto c = p.con_begin(); c != p.con_end(); ++c) {
			os << lp::fmap.get_data(c->first.first) << "/" << c->first.second << ":\n";
			const bool is_constraint = (c->first == sign_t(if_id,1));
			if (is_constraint) {
				for (const clause& cl : c->second) {
					os << "  " << decompile(cl) << ".\n";
				}
			} else {
				for (const clause& cl : c->second) {
					os << "  " << decompile(c->first,cl) << ".\n";
				}
			}
		}
		return os;
	}

	// Interactive Queries
	int Program::query(const Functor& q, const std::set<id_type>& v, deadline_t* deadline)
	{
		assert(q.sign(if_id,1));
		clause qcl;
		make_wam_instructions(q,v,true,&qcl); // inserts ?/1
		assert(qcl.back().opcode == instruction::PROMPT);
		//std::cerr << "Query instructions:\n"; for (auto& i : qcl) std::cerr << "  " << i << "\n";
		// Execute Instructions (consult/1 may create more queries)
		lp::qconstraints qc(params);
		qc.interactive = true;
		answer_type ans;
		return this->exec(qc,ans,&qcl,nullptr,deadline);
	}

	int Program::query(const clause& q, const std::set<id_type>& v, deadline_t* deadline) 
	{
		lp::qconstraints qc(this->params);
		qc.interactive = true;
		answer_type ans;
		return this->exec(qc,ans,&q,nullptr,deadline);
	}

	// Non-interactive queries
	bool Program::query(const Functor& q, deadline_t* deadline)
	{
		assert(q.sign(if_id,1));
		lp::qconstraints qc(this->params);
		qc.interactive = false;
		qc.recall = 1;
		answer_type a;
		return this->query(q,qc,a,deadline) > 0;
	}

	bool Program::query(const clause& q, long long* ic, deadline_t* deadline)
	{
		lp::qconstraints qc(this->params);
		qc.interactive = false;
		qc.recall = 1;
		answer_type a;
		return this->query(q,qc,a,ic,deadline) > 0;
	}

	bool Program::query(const Functor& q, lp::qconstraints qc, deadline_t* deadline)
	{
		assert(q.sign(if_id,1));
		qc.interactive = false;
		qc.recall = 1;
		answer_type a;
		return this->query(q,qc,a,deadline) > 0;
	}

	int Program::query(const Functor& q, lp::qconstraints& qc, lp::answer_type& a, deadline_t* deadline)
	{
		assert(q.sign(if_id,1));
		clause qcl;
		make_wam_instructions(q,true,&qcl);
		// Execute Instructions (consult/1 may create more queries)
		return this->exec(qc,a,&qcl,nullptr,deadline);
	}

	int Program::query(const clause& q, lp::qconstraints& qc, lp::answer_type& a, long long* ic, deadline_t* deadline)
	{
		return this->exec(qc,a,&q,ic,deadline);
	}

	//int Program::query(const Functor& q, lp::qconstraints& qc, std::vector<Functor>& b)
	//{
	//	clause qcl;
	//	make_wam_instructions(q,true,&qcl); // inserts ?/1
	//	// Execute Instructions (consult/1 may create more queries)
	//	answer_type a;
	//	return this->exec(qc,a,&b,&qcl);
	//}

	void Program::pop_back(const std::pair<id_type,int>& p)
	{
		//std::cerr << "pop_back: " << fmap.get_data(p.first) << "/" << p.second << "\n";
		auto at = kb.find(p);
		if (at != kb.end()) {
			at->second.pop_back();
			fix_pointers(at);
		}
	}

	void Program::push_front(const cl_def& def)
	{
		auto at = kb.find(def.first);
		if (at == kb.end()) {
			at = kb.insert(std::make_pair(def.first,concept(1,def.second))).first;
		} else {
			at->second.insert(at->second.begin(), def.second);
		}
		fix_pointers(at);
	}

	void Program::push_back(const cl_def& def)
	{
		auto at = kb.find(def.first);
		if (at == kb.end()) {
			at = kb.insert(std::make_pair(def.first,concept(1,def.second))).first;
		} else {
			at->second.push_back(def.second);
		}
		fix_pointers(at);
	}

	void Program::push_back(const clause& cl)
	{
		const sign_t s = sign_t(if_id,1);
		auto at = kb.find(s);
		if (at == kb.end()) {
			at = kb.insert(std::make_pair(s,concept(1,cl))).first;
		} else {
			at->second.push_back(cl);
		}
		fix_pointers(at);
	}

	bool Program::erase_concept(const sign_t& s)
	{
		auto at = kb.find(s);
		if (at == kb.end()) {
			return false;
		} else {
			kb.erase(at);
			return true;
		}
	}

	bool Program::covers(const Functor& cand, const Functor& ex, deadline_t* deadline)
	{
		// Insert candidate
		bool res;
		assert( cand.is_function() );
		const sign_t sc = cand.head()->signature();
		push_back(cand);
		Functor q(if_id);
		try {
			if (ex.head()) {
				// Positive example
				q.steal( const_cast<Functor*>(&ex) );
			} else {
				// Negative example
				// :- body => :- once(body)
				q.steal( const_cast<Functor*>(&ex)->body() );
			}
			res = query(q,deadline);
		} catch (...) {
			// Erase candidate before rethrowing
			q.arg_release();
			this->pop_back(sc);
			throw;
		}
		// Erase candidate
		q.arg_release();
		this->pop_back(sc);
		return res;
	}

	bool Program::covers(const Functor& cand, const clause& cl, deadline_t* deadline)
	{
		// Insert candidate
		bool res;
		assert( cand.is_function() && cand.head()->is_function() );
		const sign_t sc = cand.head()->signature();
	    //std::cerr << "Covers, adding candidate: " << *cand << "\n";
		push_back(cand);
		try {
			//std::cerr << "  evaluating:\n";
			//for (auto i = cl.begin(); i != cl.end(); ++i) cerr << "  "<< *i << "\n";
			res = query(cl,nullptr,deadline);
		} catch (...) {
			// Erase candidate before rethrowing
			this->pop_back(sc);
			throw;
		}
		// Erase candidate
		this->pop_back(sc);
		return res;
	}

	bool Program::covers(const clause& cl, long long* ic, double* time, deadline_t* deadline)
	{
		const auto param_reorder_negs = params.force_int(parameters::reorder_negs);

		std::clock_t start_time,end_time;

		if (time) {
			start_time = std::clock();
		}
		const bool res = this->query(cl,ic,deadline);
		if (time) {
			end_time = std::clock();
			*time = ((end_time-start_time) / double(CLOCKS_PER_SEC));
			//std::cerr << "Time: " << *time << "\n";
		}
		return res;
	}

	std::vector<std::pair<sign_t,stat>> Program::generalize(const char* s)
	{
		std::vector<std::pair<sign_t,stat>> res;
		std::set<sign_t> signs;
		for (;;) {
			auto at = std::find_if(modev.begin(),modev.end(),[](const Mode& m){ return m.is_head(); });
			if (at == modev.end()) break;
			const Mode& m = *at;

			const sign_t sign = m.atom().signature();
			if (signs.find(sign) == signs.end()) {
				stat r;
				if (std::strcmp(s,"nrsample") == 0) {
					r = generalize_nrsample(sign);
				} else if (std::strcmp(s,"id_nrsample") == 0) {
					r = generalize_id_nrsample(sign);
				} else if (std::strcmp(s,"emulate_nrsample") == 0) {
					r = generalize_emulated_nrsample(sign);
				} else if (std::strcmp(s,"enumerate") == 0) {
					r = generalize_enumerate(sign);
				} else if (std::strcmp(s,"heuristic") == 0) {
					r = generalize_best_first(sign);
				} else {
					throw invalid_name(s + std::string(" is not a generalization method"));
				}
				// Insert signature
				signs.insert(sign);
				res.push_back(std::make_pair(std::move(sign),std::move(r)));
			}
		}
		return res;
	}

	//stat Program::generalize(id_type method)
	//{
	//	switch (method) {
	//	case nrsample_id: 
	//		//if (params.is_set(parameters::posonly)) { 
	//		//	return generalize_nrsample_posonly();
	//		//} else {
	//			return generalize_nrsample();
	//		//}
	//	//case heuristic_id: 
	//	//	if (params.is_set(parameters::posonly)) {
	//	//		DEBUG_WARNING( cerr << "warning: best-first cannot be used with positive only examples\n" );
	//	//		throw invalid_name("best-first does not support positive data only learning");
	//	//	} else {
	//	//		return generalize_best_first(); 
	//	//	}
	//	//case enumerate_id:
	//	//	if (params.is_set(parameters::posonly)) {
	//	//		DEBUG_WARNING( cerr << "warning: enumerate cannot be used with positive only examples\n" );
	//	//		throw invalid_name("enumerate does not support positive data only learning");
	//	//	} else {
	//	//		return generalize_enumerate(); 
	//	//	}
	//	default: {
	//		stringstream ss;
	//		ss << "warning: invalid argument " << fmap.get_data(method);
	//		throw invalid_name(ss.str().c_str());
	//			 }
	//	}
	//}

	void Program::read(std::istream& is)
	{
		// Read file
		long long linec = 1; // keep track of line number (for error messages)
		while (is && !is.eof()) {
			// Single consult
			std::unique_ptr<Functor> f( lp::stream_to_expr(is,linec) );
			if (!f) break; // end of reading
			//std::cerr << "consult, read: " << *f << " (top: " << Functor::get_data(f->id()) << "/" << f->arity() << "\n";
			if (f->is_function(query_id,1)) {
				// Perform query (presumably with side effects)
				Functor q(if_id,new Functor(std::move(*f->arg_first())));
				qconstraints qc(params);
				qc.interactive = false;
				query(q,qc);
			} else if (f->is_function(qmark_id,1) && f->arg_first()->is_function(if_id,1)) {
				// Progol style query
				//std::cerr << "  progol style query: " << *f->arg_first() << "\n";
				qconstraints qc(params);
				qc.interactive = false;
				query(*f->arg_first(),qc);
				//std::cerr << "  finished query\n";
			} else {
				// add clause
				compile_insert(*f,false);
			}
		}
		//std::cout << "Read file " << filename << "\n";
	}

	bool Program::check_type(const Functor& f, id_type type)
	{
		//cerr << "Check type: " << *f << " of type " << fmap.get_data(type) << "\n";
		if (type == any_id) return true;
		if (type == nat_id) return f.is_int() && f.to_int() >= 0;

		// Query: type(f)
		Functor t(type);
		t.attach_child(const_cast<Functor*>(&f));
		Functor q(if_id);
		q.attach_child(&t);
		bool res;
		try {
			res = this->query(q);
		} catch (...) {
			q.unattach_child();
			t.unattach_child();
			throw;
		}
		q.unattach_child();
		t.unattach_child();
		//std::cerr << "Cap: " << *cap << "\n";
		return res;
	}


	wam_state Program::wam_reset(const clause* eq_ptr)
	{
		// Backup
		wam_state ws;
		ws.cap = cap;
		ws.cap2 = cap2;
		ws.heap_i = heap_i;
		ws.heap_back = heap_back;
		ws.env_i = env_i;
		ws.env_back = env_back;
		ws.env_back0 = env_back0;
		ws.stack_bottom = stack_bottom;
		ws.tindex = tindex;
		ws.num_of_args = num_of_args;
		ws.var_map = var_map;
		ws.arg_map = arg_map;
		ws.qlevel = qlevel;
		ws.qlevel2 = qlevel2;
		ws.cwd = CURRENT_PATH;

		// Reset
		cap.reset(const_cast<clause*>(eq_ptr));
		cap2 = cap;
		// Advance heap
		++heap_i;
		heap_back = heap_i;
		// Advance stack
		++env_i;
		env_back = env_i;
		env_back0 = env_i;
		stack_bottom = env_i;
		// Advance trail
		++tindex;
		num_of_args = 0;
		// Runtime info
		qlevel = 0;
		qlevel2 = 0;
		// Note: do NOT change CURRENT_PATH! cwd = PATH_TO_PROGRAM;

		var_map.clear();
		arg_map.clear();

		return ws;
	}

	void Program::wam_reset(const wam_state& ws)
	{
		cap = ws.cap;
		cap2 = ws.cap2;
		heap_i = ws.heap_i;
		heap_back = ws.heap_back;
		env_i = ws.env_i;
		env_back = ws.env_back;
		env_back0 = ws.env_back0;
		stack_bottom = ws.stack_bottom;
		num_of_args = ws.num_of_args;
		tindex = ws.tindex;
		qlevel = ws.qlevel;
		qlevel2 = ws.qlevel2;
		CURRENT_PATH = ws.cwd;
		var_map = ws.var_map;
		arg_map = ws.arg_map;
	}

	void Program::compile_insert(
		const Functor& cl,
		const std::set<id_type>& qvars,
		bool collect,
		bool at_end)
	{
		clause* ci;
		// Insert try_me_else or retry_me_else
		if (!cl.head()) {
			if (cl.body_size() == 1) {
				// Negative example
				const auto p = std::make_pair(if_id,1);
				auto at = kb.find(p);
				if (at == kb.end()) {
					at = kb.insert(std::make_pair(p,concept(1))).first;
					//std::cerr << "Make_wam_instr, inserted new: " << fmap.get_data(at->first.first) 
					//	<< "/" << at->first.second << "\n";
					ci = &at->second.front();
				} else {
					if (at_end) {
						at->second.push_back(clause());
						ci = &at->second.back();
					} else {
						at->second.insert(at->second.begin(),clause());
						ci = &at->second.front();
					}
				}
			} else {
				std::cerr << "Negative example with multiple goals not supported\n";
				std::cerr << "If this was meant to be a query, call with qmode=true\n";
				assert(false);
				exit(1);
			}
		} else {
			// Fact or Rule
			const sign_t p = cl.head()->signature();
			//std::cerr << "Signature: " << fmap.get_data(p.first) << "/" << p.second << "\n";
			auto at = kb.find(p);
			if (at == kb.end()) {
				// First occurrence: try_me_else
				//std::cerr << "cl.head() id: " << lp::fmap.get_data(p.first) << "\n"; 
				//std::cerr << "code_map, inserting: (" << p.first << "," << p.second << ") -> " << curr_line << "\n";
				at = kb.insert(std::make_pair(p,concept(1))).first;
				ci = &at->second.front();
				ci->push_back(instruction(instruction::NOOP)); // insert space for try_me_else in future
			} else {
				if (at_end) {
					// Change previous TRUST_ME to RETRY_ME_ELSE
					//std::cerr << "p: " << fmap.get_data(p.first) << "/" << p.second << " (" << at->second.size() << ")\n";
					clause& last_clause = at->second.back();
					instruction& pi = last_clause[0];
					if (at->second.size() == 1) {
						//std::cerr << "Should be NOOP: " << pi << "\n";
						assert( pi.opcode == instruction::NOOP );
						pi = instruction(instruction::TRY_ME_ELSE,p.first);
					} else {
						assert(pi.opcode == instruction::TRUST_ME);
						pi = instruction(instruction::RETRY_ME_ELSE, int(at->second.size()));
					}
					// Push back this choice point
					at->second.push_back(clause());
					ci = &at->second.back();
					ci->push_back(instruction(instruction::TRUST_ME));
				} else {
					// Change previous TRY_ME to RETRY_ME_ELSE
					clause& first_clause = at->second.front();
					instruction& pi = first_clause[0];
					if ( at->second.size() == 1 ) {
						assert( pi.opcode == instruction::NOOP );
						pi = instruction(instruction::TRUST_ME);
					} else {
						assert( pi.opcode == instruction::TRY_ME_ELSE );
						pi = instruction(instruction::RETRY_ME_ELSE,2);
					}
					// Push front this choice point
					at->second.insert( at->second.begin(), clause() );
					ci = &at->second.front();
					ci->push_back(instruction(instruction::TRY_ME_ELSE,p.first));
				}
			}
		}

		return make_wam_instructions(cl,qvars,collect,ci);
	}


	// Execute WAM (in interactive/variable binding mode)
	int Program::exec(
		const lp::qconstraints& qc, 
		answer_type& ans,
		const clause* eq_ptr,
		long long* ic, // instruction counter
		deadline_t* deadline)
	{
		//std::cerr << "Qdepth: " << qc.depth << ", Recall: " << qc.recall << "\n";
		int resc = 0; // resolution counter
		int solc = 0; // solution counter
		if (ic) *ic = 0; // instruction counter
		bool finished = false;
		wam_state ws = wam_reset(eq_ptr);
		//std::cerr << "exec, env_back: " << env_back << ", stack_bottom: " << stack_bottom << "\n";
		try {
			for ( ; !finished && resc < qc.resolutions; ) {
				if (deadline) {
					if (std::chrono::steady_clock::now() > *deadline) {
						DEBUG_WARNING(std::cerr << "WAM Execution Timed Out\n");
						throw wam_fail();
					}
				}
				if (ic) {
					++(*ic); // increase counter
					if (*ic > qc.instructions) break;
				}
				const instruction& i = *cap;
				//std::cerr << "instr: " << i << ", resc: " << resc << "\n";
				switch (i.opcode) {
				case instruction::STOP: finished = true; break;
				case instruction::NOOP: noop(); break;
				case instruction::PROCEED: proceed(); break;
				case instruction::CALL:
					++resc; 
					if (qlevel >= qc.depth) backtrack();
					else { 
						call(i.a1,i.a2,i.a3); 
					}
					break;
				case instruction::EXEC: 
					++resc;
					if (qlevel >= qc.depth) backtrack();
					else {
						execute(i.a1,i.a2);
					}
					break;
				case instruction::DYN_CALL:
					++resc;
					if (qlevel >= qc.depth) backtrack();
					else {
						dynamic_call(i.a1,i.a2); 
					}
					break;
				case instruction::DYN_EXEC: 
					++resc;
					if (qlevel >= qc.depth) backtrack();
					else {
						dynamic_execute(i.a1);
					}
					break;
				case instruction::DYN_STACK_CALL:
					++resc;
					if (qlevel >= qc.depth) backtrack();
					else {
						dynamic_stack_call(i.a1,i.a2); 
					}
					break;
				case instruction::DYN_STACK_EXEC: 
					++resc;
					if (qlevel >= qc.depth) backtrack();
					else {
						dynamic_stack_execute(i.a1);
					}
					break;
				case instruction::PUT_STR: put_structure(i.a1,i.a2,i.a3); break;
				case instruction::PUT_LIS: put_list(i.a1); break;
				case instruction::SET_VAR: set_variable(i.a1); break;
				case instruction::SET_VARY: set_stack_variable(i.a1); break;
				case instruction::SET_VAL: set_value(i.a1); break;
				case instruction::SET_VALY: set_stack_value(i.a1); break;
				case instruction::GET_STR: get_structure(i.a1,i.a2,i.a3); break;
				case instruction::GET_LIS: get_list(i.a1); break;
				case instruction::UNI_VAR: unify_variable(i.a1); break;
				case instruction::UNI_VARY: unify_stack_variable(i.a1); break;
				case instruction::UNI_VAL: unify_value(i.a1); break;
				case instruction::UNI_VALY: unify_stack_value(i.a1); break;
				case instruction::PUT_VAR: put_variable(i.a1,i.a2); break;
				case instruction::PUT_VAL: put_value(i.a1,i.a2); break;
				case instruction::GET_VAR: get_variable(i.a1,i.a2); break;
				case instruction::GET_VAL: get_value(i.a1,i.a2); break;
				case instruction::GET_VALY: get_stack_value(i.a1,i.a2); break;
				case instruction::ALLOC: allocate(); break;
				case instruction::DEALLOC: deallocate(); break;
				case instruction::GLOBALIZE_SVARS: globalize_svars(); break;
				case instruction::GET_VARY: get_stack_variable(i.a1,i.a2); break;
				case instruction::PUT_VARY: put_stack_variable(i.a1,i.a2); break;
				case instruction::PUT_VALY: put_stack_value(i.a1,i.a2); break;
				case instruction::TRY_ME_ELSE: ++resc; try_me_else(i.a1); break;
				case instruction::RETRY_ME_ELSE: ++resc; retry_me_else(i.a1); break;
				case instruction::TRUST_ME: ++resc; trust_me(); break;
				case instruction::PROMPT:
					if (!prompt(qc,ans,solc)) finished = true;
					break;
				case instruction::PUT_CON: put_constant(i.a1,i.a2); break;
				case instruction::GET_CON: get_constant(i.a1,i.a2); break;
				case instruction::SET_CON: set_constant(i.a1); break;
				case instruction::UNI_CON: unify_constant(i.a1); break;
				case instruction::PUT_INT: put_int(i.i1,i.a2); break;
				case instruction::GET_INT: get_int(i.i1,i.a2); break;
				case instruction::SET_INT: set_int(i.i1); break;
				case instruction::UNI_INT: unify_int(i.i1); break;
				case instruction::PUT_FLT: put_float(i.d1,i.a2); break;
				case instruction::GET_FLT: get_float(i.d1,i.a2); break;
				case instruction::SET_FLT: set_float(i.d1); break;
				case instruction::UNI_FLT: unify_float(i.d1); break;
				case instruction::SET_VOID: set_void(i.a1); break;
				case instruction::UNI_VOID: unify_void(i.a1); break;
				case instruction::PUT_UNSAFE_VALY: put_unsafe_value(i.a1,i.a2); break;
				case instruction::SET_LOCAL_VAL: set_local_value(i.a1); break;
				case instruction::SET_LOCAL_VALY: set_local_stack_value(i.a1); break;
				case instruction::UNI_LOCAL_VAL: unify_local_value(i.a1); break;
				case instruction::UNI_LOCAL_VALY: unify_local_stack_value(i.a1); break;
				case instruction::REG_HVAR: reg_hvar(i.a1); break;
				case instruction::REG_SVAR: reg_svar(i.a1,i.a2); break;
				case instruction::REG_ARG: reg_arg(i.a1); break;
				case instruction::NECK_CUT: neck_cut(); break;
				case instruction::GET_LVL: get_level(i.a1); break;
				case instruction::CUT: ++resc; cut(i.a1); break;
				// Builtins
				case instruction::FAIL: ++resc; fail(); break;
				case instruction::UNIFY: ++resc; unify_instr(i.a1,i.a2); break;
				case instruction::VAR: ++resc; var(i.a1); break;
				case instruction::NONVAR: ++resc; nonvar(i.a1); break;
				case instruction::ATOMIC: ++resc; atomic(i.a1); break;
				case instruction::ATOM: ++resc; atom(i.a1); break;
				case instruction::NUMBER: ++resc; number(i.a1); break;
				case instruction::INTEGER: ++resc; integer(i.a1); break;
				case instruction::FLOAT: ++resc; isfloat(i.a1); break;
				case instruction::NONUNIFIABLE: ++resc; nonunifiable(i.a1,i.a2); break;
				case instruction::EQUAL: ++resc; equal(i.a1,i.a2); break;
				case instruction::UNEQUAL: ++resc; unequal(i.a1,i.a2); break;
				case instruction::BEFORE: ++resc; before(i.a1,i.a2); break;
				case instruction::BEFORE_EQ: ++resc; before_eq(i.a1,i.a2); break;
				case instruction::AFTER: ++resc; after(i.a1,i.a2); break;
				case instruction::AFTER_EQ: ++resc; after_eq(i.a1,i.a2); break;
				case instruction::COMPARE: ++resc; compare(i.a1,i.a2,i.a3); break;
				case instruction::VARIANT: ++resc; variant(i.a1,i.a2); break;
				case instruction::NOT_VARIANT: ++resc; notvariant(i.a1,i.a2); break;
				case instruction::HALT: halt(); break;
				case instruction::ASSERT: ++resc; assertz(i.a1); break;
				case instruction::ASSERTA: ++resc; asserta(i.a1); break;
				case instruction::NB_LINKARG: ++resc; nb_linkarg(i.a1,i.a2,i.a3); break;
				case instruction::DUPE_TERM: ++resc; duplicate_term(i.a1,i.a2); break;
				case instruction::RETRACTALL: ++resc; retract_all(i.a1); break;
				case instruction::OP: ++resc; define_op(i.a1,i.a2,i.a3); break;
				case instruction::LISTING: ++resc; listing(true); break;
				case instruction::LISTING_WAM: ++resc; listing(false); break;
				case instruction::LISTING1: ++resc; listing(i.a1,true); break;
				case instruction::LISTING_WAM1: ++resc; listing(i.a1,false); break;
				case instruction::WRITE: ++resc; write(i.a1); break;
				case instruction::NL: ++resc; nl(); break;
				case instruction::CONSULT: ++resc; consult(i.a1); break;
				case instruction::CONSULT_LIST: ++resc; consult_list(i.a1,i.a2); break;
				case instruction::IS: ++resc; is(i.a1,i.a2); break;
				case instruction::NUMEQUAL: ++resc; numequal(i.a1,i.a2); break;
				case instruction::NUMUNEQUAL: ++resc; numunequal(i.a1,i.a2); break;
				case instruction::NUMLESS: ++resc; numless(i.a1,i.a2); break;
				case instruction::NUMLESSEQ: ++resc; numlesseq(i.a1,i.a2); break;
				case instruction::NUMGREATER: ++resc; numgreater(i.a1,i.a2); break;
				case instruction::NUMGREATEREQ: ++resc; numgreatereq(i.a1,i.a2); break;
				case instruction::UNIV:	++resc; univ(i.a1,i.a2); break;
				case instruction::FUNCTOR: ++resc; functor(i.a1,i.a2,i.a3); break;
				case instruction::ARG: ++resc; arg(i.a1,i.a2,i.a3); break;
				case instruction::NAME: ++resc; name(i.a1,i.a2); break;
				case instruction::MODEH: ++resc; modeh(i.a1,i.a2); break;
				case instruction::MODEB: ++resc; modeb(i.a1,i.a2,-1); break;
				case instruction::MODEB3: ++resc; modeb(i.a1,i.a2,i.a3); break;
				case instruction::IMPLIES: ++resc; add_constraint(i.a1,i.a2,false); break;
				case instruction::PREVENTS: ++resc; add_constraint(i.a1,i.a2,true); break;
				case instruction::PREVENT: ++resc; add_constraint(-1,i.a1,true); break;
				case instruction::SET_PARAM: ++resc; set_param(i.a1,i.a2); break;
				case instruction::GET_PARAM: ++resc; get_param(i.a1,i.a2); break;
				case instruction::DETERMINATION: ++resc; determination(i.a1,i.a2,true); break;
				case instruction::EXCLUDE_PRED: ++resc; determination(i.a1,i.a2,false); break;
				case instruction::GENERALIZE0: ++resc; generalize(); break;
				case instruction::GENERALIZE1: ++resc; generalize(i.a1); break;
				default:
					std::cerr << "Error: unrecognized instruction: " << i.opcode << "\n";
					assert(false);
					std::exit(1);
				}
			}

			//if (resc >= qc.resolutions) {
			//	std::cerr << "REACHED RESC LIMIT: " << resc << " = " << qc.resolutions << "\n";
			//}
		} catch (wam_fail) {
			//std::cout << "WAM: no more solutions available\n";
			// Fall through
		} catch (...) {
			wam_reset(ws); // restore WAM state
			throw;
		}

		// Restore WAM state
		wam_reset(ws);
		return solc;
	}

	void Program::detect_reflexivity(
		const std::list<Functor>& posv, 
		const std::list<Functor>& negv,
		const std::map<const Functor*,const clause*>& fun2cl,
		Constraints& constr)
	{
		//constr.reflexive.clear();
		const int param_detect_reflexivity = params.force_int(parameters::detect_reflexivity);
		if (param_detect_reflexivity > 0) {
			// First, try to disprove reflexivity by checking negative examples
			std::set<sign_t> signs;
			for (const auto& m : modev) {
				if (!m.is_head()) continue;
				const sign_t s = m.atom().signature();
				if (signs.find(s) != signs.end()) continue; // already done
				signs.insert(s);
				// Make new variable for all inputs/outputs
				Functor q = m.prototype();
				const id_type id = Functor::unique_index();
				int argc = 0;
				for (auto i = m.begin(); i != m.end(); ++i) {
					if (i->iotype == Mode::ground) continue;
					*q.get(i->pos) = Functor(id);
					++argc;
				} // for each +/- type
				if (argc < 2) continue; // no reflexivity if we have less than two variables
				// Try to prove non-reflexivity
				int rc = 0; // -1 means "not reflexive", 0+ are counts
				//for (const auto& n : negv) {
				//	if (n.arg_last()->signature() != s) continue;
				//	if (lp::unify(&q,n.arg_last())) {
				//		rc = -1;
				//		std::cerr << "Detected nonreflexivity: " << n << "\n";
				//		break;
				//	}
				//} // for each negative example
				// Inductively count how many reflexive instances we have
				if (rc == 0) {
					for (const auto& p : posv) {
						if (p.signature() != s) continue;
						if (lp::unify(&q,&p)) {
							DEBUG_INFO(std::cout << "Reflexive instance: " << p << "\n");
							constr.uncond.insert(std::make_pair(p,implication(q,true)));
							++rc;
						}
					} // for each negative example
				}
				//// Add reflexivity constraint
				//if (rc >= 0 && rc >= param_detect_reflexivity) {
				//	// add query: prevent
				//	std::cerr << "Adding: prevent " << q << "\n";
				//	this->block.push_back(implication(std::vector<Functor>(),std::move(q),true));
				//}
				//constr.reflexive.insert(std::make_pair(s,rc));
				//std::cerr << "Modeh: " << m.atom() << ", reflexivity: " << rc << "\n";
			} // for each mode
		}
	}


	void Program::detect_symmetry(
		const std::list<Functor>& posv, 
		const std::list<Functor>& negv,
		const std::map<const Functor*,const clause*>& fun2cl,
		Constraints& constr)
	{
		//constr.symmetric.clear();
		const int param_detect_symmetry = params.force_int(parameters::detect_symmetry);
		if (param_detect_symmetry > 0) {
			// First, try to disprove symmetry by checking negative examples
			// p(X,Y) and not p(Y,X) => no symmetry
			// Try on first two input variables
			for (const auto& m : modev) {
				if (!m.is_head()) continue;
				std::pair<Functor,Functor> sp;
				const sign_t s = m.atom().signature();
				// Make new variable for all inputs/outputs			
				sp.first = m.prototype();
				sp.second = m.prototype();
				const id_type v1 = Functor::unique_index();
				const id_type v2 = Functor::unique_index();
				int argc = 0;
				for (auto i = m.input_begin(); i != m.input_end(); ++i) {
					++argc;
					if (argc == 1) {
						*sp.first.get(i->pos) = Functor(v1);
						*sp.second.get(i->pos) = Functor(v2);
					} else if (argc == 2) {
						*sp.first.get(i->pos) = Functor(v2);
						*sp.second.get(i->pos) = Functor(v1);
					} else {
						// All other input variables are fixed
						*sp.first.get(i->pos) = *sp.second.get(i->pos) = Functor(Functor::unique_index());
					}
				} // for each input type
				// Handle output variables
				bool try_idempotent = false;
				if (argc == 0) {
					continue;
				}
				auto j = m.output_begin();
				if (argc == 1) {
					// If we only had one input variable, check for idempotence
					if (j == m.output_end()) {
						continue; // cannot be symmetric
					}
					*sp.first.get(j->pos) = Functor(v2);
					*sp.second.get(j->pos) = Functor(v1);
					try_idempotent = true;
					++j;
				}
				// All (other) output variables must be the same
				for ( ; j != m.output_end(); ++j) {
					*sp.first.get(j->pos) = *sp.second.get(j->pos) = Functor(Functor::unique_index());
				}
				
				// Try to prove non-symmetry
				//int sc = 0; // -1 means "not symmetric", 0+ are counts
				//for (const auto& n : negv) {
				//	Subst subs;
				//	if (lp::unify(&q1,n.arg_last(),subs)) {
				//		for (const auto& p : posv) {
				//			if (lp::unify(&q2,&p,subs)) {
				//				// disproof
				//				std::cerr << "Detected nonsymmetry: " << n << " and " << p << "\n";
				//				sc = -1;
				//				goto next_mode;
				//			}
				//		}
				//		break;
				//	}
				//} // for each negative example
				// Inductively count how many symmetric instances we have
				for (auto i = posv.begin(); i != posv.end(); ++i) {
					Subst subs;
					if (lp::unify(&sp.first,&*i,subs)) {
						for (auto j = std::next(i); j != posv.end(); ++j) {
							if (lp::unify(&sp.second,&*j,subs)) {
								// TODO: handle idempotent case ==> p(_,Out) -> p(Out,_)
								DEBUG_INFO(std::cout << "Symmetric instance: " << *i << "  <==>  " << *j << "\n");
								constr.uncond.insert(std::make_pair(*i,implication(sp.first,sp.second,true)));
								constr.uncond.insert(std::make_pair(*j,implication(sp.first,sp.second,true)));
								//++sc;
							}
						}
					}
				}
//				// Add symmetry constraint
//				if (sc >= 0 && sc >= param_detect_symmetry) {
//					// add query: prevent
//					std::cerr << "Adding (symmetry): " << q1 << " prevents " << q2 << "\n";
//					this->block.push_back(implication(std::vector<Functor>(1,q1),std::move(q2),true));
//				}
//next_mode:
//				//constr.symmetry.insert(std::make_pair(m,sc));
//				std::cerr << "Modeh: " << m.atom() << ", symmetry: " << sc << "\n";
			} // for each mode
		}
	}

	void Program::detect_dependent(
		const std::list<Functor>& posv, 
		const std::list<Functor>& negv,
		const std::map<const Functor*,const clause*>& fun2cl,
		Constraints& constr)
	{
		constr.dependent_vars.clear();
		const bool param_detect_dependent = params.is_set(parameters::detect_dependent);
		if (param_detect_dependent) {
			/* 
			// Note: dummies only apply to input variables
			1st arg is Dummy var => 
			R(a,REST) = R'(REST)
			R(b,REST) = R'(REST)
			so R(a,REST) = R(b,REST)
			So if we can find R(a,REST) != R(b,REST), then var is not dummy
			*/
			// Return index of argument that is not equal (0..n), or negative on failure
			// Store all dummy variables
			// For each modeh, check if any input variable is dummy
			for (auto& m : modev) {
				if (!m.is_head()) continue;
				// Check if this input variable is non-dummy
				const sign_t s = m.atom().signature();
				for (auto i = m.input_begin(); i != m.input_end(); ++i) {
					// TODO: use hashes
					for (const auto& n : negv) {
						if (n.arg_last()->signature() == s) {
							for (const auto& p : posv) {
								if (p.signature() != s) continue;
								if (!equal_except(i->pos,n.arg_last(),&p)) {
									//DEBUG_INFO(std::cout << "Input Variable at position " << i->pos << " is dependent (" << m << ")\n");
									constr.dependent_vars.insert(std::make_pair(m,i->pos));
									DEBUG_INFO(std::cout << "Detected variable dependency in: " << p << "\n");
									// TODO: make dependent variables example-specific?
									//++constr.dependent[p];
									goto next_inputvar;
								}
							} // for each positive example
						}
					} // for each negative example
next_inputvar: ;
				} // for each input variable
			} // for each mode
		}
	}


	void Program::post_detect_properties(const std::list<Functor>& posv, const std::list<Functor>& negv)
	{
		const int sample_size = params.force_int(parameters::post_detect_samples);

		DEBUG_BASIC(std::cerr << "post_detect_propeties, posv: " << posv.size() << "\n");
		if (posv.empty()) return;
		const sign_t sign = posv.front().signature();
		//std::cerr << "signature: " << Functor::get_data(sign.first) << "/" << sign.second << "\n";

		// Collect ground terms
		typedef std::multimap<id_type,std::set<Functor>> type2gterms;
		typedef std::map<id_type,type2gterms> mode2tg;
		mode2tg m2tg;
		for (const auto& m : modev) {
			if (!m.is_head() && m.atom().signature() == sign) {
				//std::cerr << "  mode: " << m << "\n";
				type2gterms t2g;
				std::for_each(posv.begin(),posv.end(),[&](const Functor& ex){
					std::for_each(m.input_begin(),m.input_end(),[&](const Mode::pmarker& pm){
						auto at = t2g.find(pm.type);
						if (at == t2g.end()) at = t2g.insert(std::make_pair(pm.type,std::set<Functor>()));
						at->second.insert(*ex.get(pm.pos));
					});
				});
				m2tg.insert(std::make_pair(m,std::move(t2g)));
			}
		}

		auto random_sample = [&](const Mode& m, id_type type) -> const Functor& {
			const auto& t2g = m2tg.find(m)->second;
			const auto& gs = t2g.find(type)->second;
			assert(!gs.empty());
			auto i = gs.begin();
			std::advance(i, rand() % gs.size());
			return *i;
		};

		auto make_query = [&](Functor& ex) -> bool {
			bool ans;
			Functor q(if_id);
			q.attach_child(&ex);
			try {
				ans = this->query(q);
			} catch (...) {
				q.unattach_child();
				throw;
			}
			q.unattach_child();
			return ans;
		};

		// Check reflexivity
		for (auto& m : modev) {
			if (!m.is_head() && m.atom().signature() == sign && m.props[Mode::reflexive] == 0) {
				//std::cerr << "reflexivity, mode: " << m << "\n";
				const type2gterms& t2g = m2tg.find(m)->second;
				// Initialize common_types
				std::set<Functor> common_types;
				if (m.input_begin() != m.input_end()) common_types = t2g.find(m.input_begin()->type)->second;
				else if (m.output_begin() != m.output_end()) common_types = t2g.find(m.output_begin()->type)->second;
				else continue; // cannot be reflexive
				// Common types is joint of all types
				for (const auto& pm : m) {
					const auto& gs = t2g.find(pm.type)->second;
					std::set<Functor> ns;
					std::set_intersection(common_types.begin(),common_types.end(),gs.begin(),gs.end(),std::inserter(ns,ns.begin()));
					common_types = std::move(ns);
					break;
				}
				if (common_types.size() < 2) continue; // cannot be reflexive
				bool reflexive = true;
				for (int k = 0; k < sample_size; ++k) {
					// Construct functor
					Functor ftest = m.prototype();
					auto i = common_types.begin();
					std::advance(i, rand() % common_types.size());
					const Functor ginst = *i;
					std::for_each(m.begin(),m.end(),[&](const Mode::pmarker& pm){ *ftest.get(pm.pos) = ginst; });
					//std::cerr << "Testing for reflexivity: " << ftest << "\n";
					// Test functor
					if (!make_query(ftest)) {
						//std::cerr << "Not reflexive: " << ftest << " (" << m << ")\n";
						reflexive = false;
						break;
					}
				}
				if (reflexive) {
					DEBUG_BASIC(std::cerr << "Detected reflexivity for " << m << "\n");
					m.props[Mode::reflexive] = 1LL;
				}
			}
		}

		// Check symmetry
		for (auto& m : modev) {
			if (!m.is_head() && m.atom().signature() == sign && m.props[Mode::symmetric] == 0 && std::distance(m.input_begin(),m.input_end()) >= 2) {
				//std::cerr << "symmetry, mode: " << m << "\n";
				bool symmetric = true;
				const type2gterms& t2g = m2tg.find(m)->second;
				for (int k = 0; k < sample_size; ++k) {
					// Construct functor
					Functor ftest = m.prototype();
					std::pair<Functor::position,Functor::position> in;
					int in_counter = 0;
					for (const auto& pm : m) {
						switch (pm.iotype) {
						case Mode::input:
							{
								Functor f = random_sample(m,pm.type); 
								switch (in_counter) {
								case 0: in.first = pm.pos; break;
								case 1: in.second = pm.pos; break;
								}
								*ftest.get(pm.pos) = std::move(f); 
								++in_counter;
								break;
							}
						case Mode::ground: 
							*ftest.get(pm.pos) = random_sample(m,pm.type); 
							break;
						}
					}
					// Test functor
					//std::cerr << "  testing symmetry: " << ftest << "\n";
					if (make_query(ftest)) {
						// First succeeds, second fails?
						std::swap(*ftest.get(in.first), *ftest.get(in.second));
						//std::cerr << "  query succeded, testing: " << ftest << "\n";
						if (!make_query(ftest)) {
							//std::cerr << "  not symmetric: failed " << ftest << "\n";
							symmetric = false;
							break;
						}
					} else {
						// First failed, second succeeds?
						std::swap(*ftest.get(in.first), *ftest.get(in.second));
						//std::cerr << "  query failed, testing: " << ftest << "\n";
						if (make_query(ftest)) {
							//std::cerr << "  not symmetric: succeeded " << ftest << "\n";
							symmetric = false;
							break;
						}
					}
					//std::cerr << "  continuing\n";
				}
				if (symmetric) {
					DEBUG_BASIC(std::cerr << "Detected symmetry for " << m << "\n");
					m.props[Mode::symmetric] = 1LL;
				}
			}
		}

		// Check idempotence
		for (auto& m : modev) {
			if (!m.is_head() && m.atom().signature() == sign && m.props[Mode::idempotent] == 0 
				&& std::distance(m.input_begin(),m.input_end()) >= 1 && std::distance(m.output_begin(),m.output_end()) >= 1) 
			{
				//std::cerr << "idempotent, mode: " << m << "\n";
				bool idempotence = true;
				const type2gterms& t2g = m2tg.find(m)->second;
				for (int k = 0; k < sample_size; ++k) {
					// Construct functor
					Functor ftest = m.prototype();
					Functor::position in_pos,out_pos;
					int in_counter = 0;
					int out_counter = 0;
					for (const auto& pm : m) {
						switch (pm.iotype) {
						case Mode::input:
							{
								Functor f = random_sample(m,pm.type); 
								if (in_counter == 0) in_pos = pm.pos;
								*ftest.get(pm.pos) = std::move(f); 
								++in_counter;
								break;
							}
						case Mode::output:
							{
								if (out_counter == 0) out_pos = pm.pos;
								++out_counter;
								break;
							}
						case Mode::ground: 
							*ftest.get(pm.pos) = random_sample(m,pm.type); 
							break;
						}
					}
					// Test functor
					//std::cerr << "  testing idempotence: " << ftest << "\n";
					if (make_query(ftest)) {
						// First succeeds, f(X) -> Y, check if f(Y) -> X
						std::swap(*ftest.get(in_pos), *ftest.get(out_pos));
						//std::cerr << "  query succeded, testing: " << ftest << "\n";
						if (!make_query(ftest)) {
							//std::cerr << "  not symmetric: failed " << ftest << "\n";
							idempotence = false;
							break;
						}
					}
					//std::cerr << "  continuing\n";
				}
				if (idempotence) {
					DEBUG_BASIC(std::cerr << "Detected idempotence for " << m << "\n");
					m.props[Mode::idempotent] = 1LL;
				}
			}
		}

	}


	void Program::update_dynamic_constraints(const Functor& ex, const Constraints& constr)
	{
		// Clear Dynamic Constraints
		for (auto i = block.begin(); i != block.end(); ) {
			if (i->is_dynamic) i = block.erase(i);
			else ++i;
		}
		for (auto i = imp.begin(); i != imp.end(); ) {
			if (i->is_dynamic) i = imp.erase(i);
			else ++i;
		}

		// Make new Dynamic Constraints
		auto uat = constr.uncond.find(ex);
		if (uat != constr.uncond.end()) {
			block.push_back(uat->second);
		}
		auto cat = constr.cond.find(ex);
		if (cat != constr.cond.end()) {
			imp.push_back(cat->second);
		}
	}

//	void Program::update_dynamic_constraints(
//		const std::list<clause>& pex, 
//		const std::list<clause>& nex,
//		Constraints& constr)
//	{
//		// Clear all dynamic constraints
//		for (auto& i : block) {
//			if (i.is_dynamic) i.clear();
//		}
//		for (auto& i : imp) {
//			if (i.is_dynamic) i.clear();
//		}
//
//		// Convert examples into Functors
//		std::vector<Functor> posv,negv;
//		posv.reserve(pex.size());
//		for (const auto& cl : pex) {
//			// Note: posexs are in form :- p(a,b)
//			Functor p = decompile(cl);
//			posv.push_back(std::move(*p.arg_last()));
//		}
//		negv.reserve(nex.size());
//		for (const auto& cl : nex) {
//			negv.push_back(decompile(cl));
//		}
//
//		// Scan examples for symmetry/transitivity/independence?
//		constr.dependent_vars.clear();
//		const bool param_detect_dependent = params.is_set(parameters::detect_dependent);
//		if (param_detect_dependent) {
//			/* 
//			// Note: dummies only apply to input variables
//			1st arg is Dummy var => 
//			R(a,REST) = R'(REST)
//			R(b,REST) = R'(REST)
//			so R(a,REST) = R(b,REST)
//			So if we can find R(a,REST) != R(b,REST), then var is not dummy
//			*/
//			// Return index of argument that is not equal (0..n), or negative on failure
//			// Store all dummy variables
//			// For each modeh, check if any input variable is dummy
//			for (auto& m : modev) {
//				if (!m.is_head()) continue;
//				// Check if this input variable is non-dummy
//				const sign_t s = m.atom().signature();
//				for (auto i = m.input_begin(); i != m.input_end(); ++i) {
//					// TODO: use hashes
//					for (const auto& n : negv) {
//						if (n.arg_last()->signature() == s) {
//							for (const auto& p : posv) {
//								if (p.signature() != s) continue;
//								if (!equal_except(i->pos,n.arg_last(),&p)) {
//									DEBUG_INFO(std::cout << "Input Variable at position " << i->pos << " is dependent (" << m << ")\n");
//									constr.dependent_vars.insert(std::make_pair(m,i->pos));
//									goto next_inputvar;
//								}
//							} // for each positive example
//						}
//					} // for each negative example
//next_inputvar: ;
//				} // for each input variable
//			} // for each mode
//		}
//
//		//constr.reflexive.clear();
//		const int param_detect_reflexivity = params.force_int(parameters::detect_reflexivity);
//		if (param_detect_reflexivity > 0) {
//			// First, try to disprove reflexivity by checking negative examples
//			std::set<sign_t> signs;
//			for (const auto& m : modev) {
//				if (!m.is_head()) continue;
//				const sign_t s = m.atom().signature();
//				if (signs.find(s) != signs.end()) continue; // already done
//				signs.insert(s);
//				// Make new variable for all inputs/outputs
//				Functor q = m.prototype();
//				const id_type id = Functor::unique_index();
//				int argc = 0;
//				for (auto i = m.begin(); i != m.end(); ++i) {
//					if (i->iotype == Mode::ground) continue;
//					*q.get(i->pos) = Functor(id);
//					++argc;
//				} // for each +/- type
//				if (argc < 2) continue; // no reflexivity if we have less than two variables
//				// Try to prove non-reflexivity
//				int rc = 0; // -1 means "not reflexive", 0+ are counts
//				for (const auto& n : negv) {
//					if (n.arg_last()->signature() != s) continue;
//					if (lp::unify(&q,n.arg_last())) {
//						rc = -1;
//						//std::cerr << "Detected nonreflexivity: " << n << "\n";
//						break;
//					}
//				} // for each negative example
//				// Inductively count how many reflexive instances we have
//				if (rc == 0) {
//					for (const auto& p : posv) {
//						if (p.signature() != s) continue;
//						if (lp::unify(&q,&p)) {
//							++rc;
//						}
//					} // for each negative example
//				}
//				// Add reflexivity constraint
//				if (rc >= 0 && rc >= param_detect_reflexivity) {
//					// add query: prevent
//					std::cerr << "Adding: prevent " << q << "\n";
//					this->block.push_back(implication(std::vector<Functor>(),std::move(q),true));
//				}
//				//constr.reflexive.insert(std::make_pair(s,rc));
//				std::cerr << "Modeh: " << m.atom() << ", reflexivity: " << rc << "\n";
//			} // for each mode
//		}
//
//		//constr.symmetry.clear();
//		const int param_detect_symmetry = params.force_int(parameters::detect_symmetry);
//		if (param_detect_symmetry > 0) {
//			// First, try to disprove symmetry by checking negative examples
//			// p(X,Y) and not p(Y,X) => no symmetry
//			// Try on first two input variables
//			for (const auto& m : modev) {
//				if (!m.is_head()) continue;
//				const sign_t s = m.atom().signature();
//				// Make new variable for all inputs/outputs
//				Functor q1 = m.prototype();
//				Functor q2 = m.prototype();
//				const id_type v1 = Functor::unique_index();
//				const id_type v2 = Functor::unique_index();
//				int argc = 0;
//				for (auto i = m.input_begin(); i != m.input_end(); ++i) {
//					++argc;
//					if (argc == 1) {
//						*q1.get(i->pos) = Functor(v1);
//						*q2.get(i->pos) = Functor(v2);
//					} else if (argc == 2) {
//						*q1.get(i->pos) = Functor(v2);
//						*q2.get(i->pos) = Functor(v1);
//					} else {
//						// All other input variables are fixed
//						*q1.get(i->pos) = *q2.get(i->pos) = Functor(Functor::unique_index());
//					}
//				} // for each input type
//				// Handle output variables
//				bool try_idempotent = false;
//				if (argc == 0) continue;
//				auto j = m.output_begin();
//				if (argc == 1) {
//					// If we only had one input variable, check for idempotence
//					if (j == m.output_end()) continue; // cannot be symmetric
//					*q1.get(j->pos) = Functor(v2);
//					*q2.get(j->pos) = Functor(v1);
//					try_idempotent = true;
//					++j;
//				}
//				// All (other) output variables must be the same
//				for ( ; j != m.output_end(); ++j) {
//					*q1.get(j->pos) = *q2.get(j->pos) = Functor(Functor::unique_index());
//				}
//				
//				// Try to prove non-symmetry
//				int sc = 0; // -1 means "not symmetric", 0+ are counts
//				for (const auto& n : negv) {
//					Subst subs;
//					if (lp::unify(&q1,n.arg_last(),subs)) {
//						for (const auto& p : posv) {
//							if (lp::unify(&q2,&p,subs)) {
//								// disproof
//								std::cerr << "Detected nonsymmetry: " << n << " and " << p << "\n";
//								sc = -1;
//								goto next_mode;
//							}
//						}
//						break;
//					}
//				} // for each negative example
//				// Inductively count how many symmetric instances we have
//				for (auto i = posv.begin(); i != posv.end(); ++i) {
//					Subst subs;
//					if (lp::unify(&q1,&*i,subs)) {
//						for (auto j = std::next(i); j != posv.end(); ++j) {
//							if (lp::unify(&q2,&*j,subs)) {
//								std::cerr << "Symmetric instance: " << *i << "  <==>  " << *j << "\n";
//								++sc;
//							}
//						}
//					}
//				}
//				// Add symmetry constraint
//				if (sc >= 0 && sc >= param_detect_symmetry) {
//					// add query: prevent
//					std::cerr << "Adding (symmetry): " << q1 << " prevents " << q2 << "\n";
//					this->block.push_back(implication(std::vector<Functor>(1,q1),std::move(q2),true));
//				}
//next_mode:
//				//constr.symmetry.insert(std::make_pair(m,sc));
//				std::cerr << "Modeh: " << m.atom() << ", symmetry: " << sc << "\n";
//			} // for each mode
//		}
//	}


}

